// RFE2 Tools
// Wes Hinsley, MRC Centre, 2010

import java.awt.Color;
import java.awt.Font;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.RenderingHints;
import java.awt.image.BufferedImage;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.BufferedReader;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.PrintWriter;
import java.util.zip.GZIPInputStream;

import javax.imageio.ImageIO;


public class ExportRFE {
  
  // A quick example to read a rain file (from within the GZ file!)
  // And put it into a Tab-separated file of values
  
  static float data[][] = new float[751][801];  // 0..751 = 20 West - 55 East.    0..801 = 40 North - 40 South
  
  final static boolean PLAIN = false;
  public static Color getCol(float sr,float sg,float sb,float er,float eg,float eb,float min,float max, float val) {
    sr/=255f;
    sg/=255f;
    sb/=255f;
    er/=255f;
    eg/=255f;
    eb/=255f;
    
    float scale = (val-min)/(max-min);
    if (scale>1) scale=1;
    float r = sr+(scale*(er-sr));
    float g = sg+(scale*(eg-sg));
    float b = sb+(scale*(eb-sb));
    return new Color(r,g,b);
  }
  
  public static Color rainColor(float val, float min, float max) {
    if (val<=0.1) return getCol(255,255,255,128,128,128,0,0.1f,val);
    else if (val<=1) return getCol(128,128,128,255,190,180,0.1f,1,val);
    else if (val<=2) return getCol(255,190,180,180,250,170,1,2,val);
    else if (val<=5) return getCol(180,250,170,120,245,115,2,5,val);
    else if (val<=10) return getCol(120,245,115,180,240,250,5,10,val);
    else if (val<=15) return getCol(180,240,250,120,185,250,10,15,val);
    else if (val<=20) return getCol(120,185,250,60,150,245,15,20,val);
    else if (val<=30) return getCol(60,150,245,30,110,235,20,30,val);
    else if (val<=40) return getCol(30,110,235,255,232,120,30,40,val);
    else if (val<=50) return getCol(255,232,120,255,160,0,40,50,val);
    else if (val<=75) return getCol(255,160,0,255,50,0,50,75,val);
    else return getCol(255,50,0,192,0,0,75,max,val);
  }
  
  
  public static void rainFrame(Graphics2D g,float min, float max, String date) {
    g.setColor(Color.BLACK);
    g.drawRect(50,35,751,801);
    g.setFont(new Font("Arial", Font.BOLD, 12));
    g.setRenderingHint(RenderingHints.KEY_ANTIALIASING,   // Anti-alias!
        RenderingHints.VALUE_ANTIALIAS_ON);

    final float[] sv = new float[] {0,0.1f,1,2,5,10,15,20,30,40,50,75};
    final float[] ev = new float[] {0.1f,1,2,5,10,15,20,30,40,50,75,max};
    final String[] svs = new String[] {"0","0.1","1","2","5","10","15","20","30","40","50","75"};    
    for (int cat=0; cat<=11; cat++) {
      for (int x=0; x<40; x++) {
        g.setColor(rainColor(sv[cat]+(((float)x/40f)*(ev[cat]-sv[cat])),min,max));
        g.drawLine(185+(cat*40)+x,865,185+(cat*40)+x,885);
      }
      g.setColor(Color.BLACK);
      g.drawLine(185+(cat*40),865,185+(cat*40),890);
      g.drawString(svs[cat],185+(cat*40),902);
    }
    g.drawRect(185,865,480,20);
    g.setFont(new Font("Arial", Font.BOLD, 16));
    g.drawString(date,50,21);
    
    
  }
  
  public static void updateFrame(float[][] data,Graphics g, float min, float max, BufferedImage bi, BufferedImage bak) {
    g.setColor(Color.WHITE);
    if (!PLAIN) g.fillRect(0,0,851,950);
    for (int j=800; j>=0; j--) {
      for (int i=0; i<751; i++) {
        if (PLAIN) bi.setRGB(i,j,bak.getRGB(i,j));
        else bi.setRGB(i+50,j+35,bak.getRGB(i,j));
      }
    }
    for (int i=0; i<751; i++) {
      for (int j=0; j<801; j++) {
       if (data[i][j]>0) {
         g.setColor(rainColor(data[i][j],min,max));
         if (PLAIN) g.drawLine(i,j,i,j);
         else g.drawLine(i+50,j+35,i+50,j+35);
       }
      }
    }
  }
  
  
  
  public static void readData(String file) {
    try {
      DataInputStream dis = new DataInputStream(new BufferedInputStream(new FileInputStream(file)));
      for (int j=800; j>=0; j--)
        for (int i=0; i<751; i++)
          data[i][j]=dis.readFloat();
      dis.close();
    } catch (Exception e) { e.printStackTrace(); }
  }
  
  public static void extractFromGZ(String zipFile, String theFile) {
    byte[] buffer = new byte[1024];
    try {
      GZIPInputStream gzip = new GZIPInputStream(new BufferedInputStream(new FileInputStream(zipFile)));
      DataOutputStream dos = new DataOutputStream(new BufferedOutputStream(new FileOutputStream(theFile)));
      int len;
      while((len = gzip.read(buffer)) >= 0) dos.write(buffer, 0, len);
      gzip.close();
      dos.close();
    } catch (Exception e) { e.printStackTrace(); }
  }
  
  public static void writeTSV(String file) {
    try {
      PrintWriter PW = new PrintWriter(new File(file));
      for (int j=0; j<=800; j++) {
        for (int i=0; i<751; i++) PW.print(data[i][j]+"\t");
        PW.println("");
      }
      PW.flush();
      PW.close();
    } catch (Exception e) { e.printStackTrace(); }
  }
  
  final static byte SERIES = 1;
  final static byte TSV = 2;
  final static byte PNG = 3;
  static boolean ACC_MONTHLY = false;
  
  
  public static void main(String[] args) throws Exception {
    
    byte[] month_length = new byte[] {31,28,31,30,31,30,31,31,30,31,30,31};
    byte mode=0;
    float lon=0,lat=0;
    int start_year=0,start_month=0,start_day=0,end_year=0,end_month=0,end_day=0;
    boolean PLAIN = false;
    String tpath="";
    String out_path="";
    boolean gotArg=false;
    for (int i=0; i<args.length; i++) {
      if (args[i].trim().length()>0) gotArg=true;
    }
    if (!gotArg) {
      System.out.println("Example Usage:");
      System.out.println("               java ExportRFE /series /lon:-15.5 /lat:25.2 /start:20050101 /end:20050228 - return sequence of values for given co-ords and time (yyyymmdd)");
      System.out.println("                                        [optional]         /accmonthly     to give accumulated value, and daily average, for each month");
      System.out.println("");
      System.out.println("               java ExportRFE /tsv /start:20050101 /end:20050228 - export files 20050101.txt..20050228.txt - a Tab-separated text file for loading into Excel");
      System.out.println("                                        [optional]         /out:directory\\     set output path");
      System.out.println("");
      System.out.println("               java ExportRFE /png /start:20050101 /end:20050228 - export files 20050101.png..20050228.png - Produce PNG images");
      System.out.println("                                        [optional]         /plain              exports plain images, no text/borders. Can be better for re-scaling later");
      System.out.println("                                        [optional]         /out:directory\\     set output path");      
      System.exit(0);
    }
    ACC_MONTHLY = false;
    
    if (args.length>0) {
      for (int i=0; i<args.length; i++) {
        if (args[i].toUpperCase().equals("/SERIES")) mode=SERIES;
        else if (args[i].toUpperCase().equals("/TSV")) mode=TSV;
        else if (args[i].toUpperCase().equals("/TPATH:")) {
          tpath=args[i].substring(6);
          if (!tpath.endsWith("\\")) tpath+="\\";
        } else if (args[i].toUpperCase().equals("/OUT:")) {
          out_path=args[i].substring(6);
          if (!out_path.endsWith("\\")) out_path+="\\";
        }
        else if (args[i].toUpperCase().equals("/PNG")) mode=PNG;
        else if (args[i].toUpperCase().equals("/PLAIN")) PLAIN=true;        
        else if (args[i].toUpperCase().startsWith("/LON:")) lon=Float.parseFloat(args[i].substring(5));
        else if (args[i].toUpperCase().startsWith("/LAT:")) lat=Float.parseFloat(args[i].substring(5));
        else if (args[i].toUpperCase().startsWith("/START:")) {
          String start = args[i].substring(7);
          start_year = Integer.parseInt(start.substring(0,4));
          start_month = Integer.parseInt(start.substring(4,6));
          start_day = Integer.parseInt(start.substring(6,8));          
        }
        else if (args[i].toUpperCase().startsWith("/END:")) {
          String end = args[i].substring(5);
          end_year = Integer.parseInt(end.substring(0,4));
          end_month = Integer.parseInt(end.substring(4,6));
          end_day = Integer.parseInt(end.substring(6,8));          
        } else if (args[i].toUpperCase().startsWith("/ACCMONTHLY")) ACC_MONTHLY=true;
        
      }
    }
    
    BufferedImage bi=null;
    BufferedImage bak=null;
    Graphics g=null;
    
    int img=0;
    if (mode==PNG) {
      if (PLAIN) bi = new BufferedImage(751,801,BufferedImage.TYPE_3BYTE_BGR);
      else bi = new BufferedImage(851,935,BufferedImage.TYPE_3BYTE_BGR);
      bak= ImageIO.read(new File(tpath+"rain_bak.png"));
      g = bi.getGraphics();
    }
    
    double accumulate=0;
    int count_days=0;
    if (mode==SERIES) {
      int year = start_year;
      // Year is >=2001
      if (year%4==0) month_length[1]=29; else month_length[1]=28;
      int month = start_month;
      int day = start_day;
      int x_index = (int) (Math.round(lon*10)+200);
      int y_index = (int) (Math.round((40-lat)*10));
      System.out.println("(lon,lat) = ("+lon+","+lat+") - using (x,y) = ("+x_index+","+y_index+")");
      boolean not_done = true;
      while  (not_done) {
        not_done = (!((year==end_year) && (month==end_month) && (day==end_day)));
        String monthString = (month<10)?"0"+month:String.valueOf(month);
        String dayString = (day<10)?"0"+day:String.valueOf(day);
        if (!new File("all_products.bin."+year+monthString+dayString+".gz").exists()) {
          System.out.println(year+monthString+dayString+"\tMISSING");
        } else {
          extractFromGZ("all_products.bin."+year+monthString+dayString+".gz",out_path+"all_products.bin."+year+monthString+dayString);
          readData(out_path+"all_products.bin."+year+monthString+dayString);
          if (ACC_MONTHLY) {
            accumulate+=data[x_index][y_index];
            count_days++;
            if ((day==month_length[month-1]) || (not_done==false)) {
              System.out.println(year+monthString+"\t"+accumulate+"\t"+accumulate/(double)count_days);
              accumulate=0;
              count_days=0;
            }
            
          } else {
            System.out.println(year+monthString+dayString+"\t"+data[x_index][y_index]);
          }
          new File(out_path+"all_products.bin."+year+monthString+dayString).delete();
        }
        day++;
        if (day>month_length[month-1]) {
          day=1;
          month++;
        }
        if (month>12) {
          month=1;
          year++;
          // Year is >=2001
          if (year%4==0) month_length[1]=29; else month_length[1]=28;
        }
      }
    }
    
    else if ((mode==TSV) || (mode==PNG)) {
      img++;
      int year = start_year;
      // Year is >=2001
      if (year%4==0) month_length[1]=29; else month_length[1]=28;
      int month = start_month;
      int day = start_day;
      boolean not_done=true;
      while (not_done) {
        not_done = (!((year==end_year) && (month==end_month) && (day==end_day)));
        String monthString = (month<10)?"0"+month:String.valueOf(month);
        String dayString = (day<10)?"0"+day:String.valueOf(day);
        if (new File("all_products.bin."+year+monthString+dayString+".gz").exists()) {
          extractFromGZ("all_products.bin."+year+monthString+dayString+".gz",out_path+"all_products.bin."+year+monthString+dayString);
          readData(out_path+"all_products.bin."+year+monthString+dayString);
          new File(out_path+"all_products.bin."+year+monthString+dayString).delete();
          if (mode==TSV) writeTSV(year+monthString+dayString+".txt");
          else if (mode==PNG) {
            updateFrame(data,g,0,75,bi,bak);
            String imgString="";
            if (img<10) imgString+="0";
            if (img<100) imgString+="0";
            if (img<1000) imgString+="0";
            imgString+=img;
            img++;
            if (!PLAIN) rainFrame((Graphics2D)g,0,75,"Rainfall (mm)    "+year+" / "+monthString+" / "+dayString);
            ImageIO.write(bi,"PNG",new File(out_path+"img_"+imgString+".png"));
            
          }
          System.out.println(year+monthString+dayString+".txt\tOK");
        } else {
          System.out.println(year+monthString+dayString+".txt\tMISSING");
        }
        day++;
        if (day>month_length[month-1]) {
          day=1;
          month++;
        }
        if (month>12) {
          month=1;
          year++;
          // Year is >=2001
          if (year%4==0) month_length[1]=29; else month_length[1]=28;
        }
      }
    }
  }
}
